"""
===============================================================================
 AnthroHeart Permanent Archive Uploader
===============================================================================

Purpose:
--------
This script cryptographically notarizes and permanently archives a master
7z file using blockchain-backed storage (Arweave).

It is designed for long-term preservation, proof of existence, and integrity,
not for speculation, NFTs, or smart contracts.

What this script does:
----------------------
1. Creates and manages its own Python virtual environment (.venv)
2. Generates a cryptographic signing key (Ed25519)
3. Hashes the target archive (SHA-256)
4. Digitally signs the hash
5. Uploads the archive to Arweave (one-time fee, permanent storage)
6. Uploads signed metadata (hash + signature + public key)
7. Outputs immutable transaction IDs and a permanent URL

Key properties:
---------------
- One-time payment (no renewals, no pinning fees)
- Immutable, tamper-evident storage
- Verifiable authorship and integrity
- Designed to be re-runnable months or years later
- No dependency on centralized platforms after upload

Requirements:
-------------
- Ubuntu / Linux
- Python 3.8+
- An Arweave wallet JSON file (funded when ready)
- Target file (e.g. AnthroHeart_Master.7z)

Important notes:
----------------
- This script does NOT activate a venv in your shell; it safely re-execs itself
  inside the venv it creates.
- Large uploads (multi-GB) require sufficient AR balance.
- This is archival / notarization tooling, not a distribution system.

Author:
-------
Thomas Sweet
Anthro Entertainment LLC

===============================================================================
"""

import os
import sys
import subprocess

VENV_DIR = ".venv"
REQUIRED = ["arweave-python", "pynacl", "tqdm"]

def in_venv():
    return sys.prefix != sys.base_prefix

def bootstrap_venv():
    if not os.path.exists(VENV_DIR):
        print("🔧 Creating virtual environment...")
        subprocess.check_call([sys.executable, "-m", "venv", VENV_DIR])

    pip = os.path.join(VENV_DIR, "bin", "pip")
    python = os.path.join(VENV_DIR, "bin", "python")

    print("📦 Installing dependencies...")
    subprocess.check_call([pip, "install", "--upgrade", "pip"])
    subprocess.check_call([pip, "install", *REQUIRED])

    print("🔁 Re-launching inside venv...")
    subprocess.check_call([python] + sys.argv)
    sys.exit(0)

if not in_venv():
    bootstrap_venv()

# ============================
# ACTUAL LOGIC STARTS HERE
# ============================

import json
import hashlib
from nacl.signing import SigningKey
from nacl.encoding import HexEncoder
from arweave.arweave_lib import Wallet, Transaction

FILE_PATH = "AnthroHeart_Master.7z"
WALLET_JSON = "arweave_wallet.json"
OUT_DIR = "archive_proof"

os.makedirs(OUT_DIR, exist_ok=True)

# 1. Generate signing key
signing_key = SigningKey.generate()
verify_key = signing_key.verify_key

private_key = signing_key.encode(encoder=HexEncoder).decode()
public_key = verify_key.encode(encoder=HexEncoder).decode()

with open(f"{OUT_DIR}/signing_private.key", "w") as f:
    f.write(private_key)

with open(f"{OUT_DIR}/signing_public.key", "w") as f:
    f.write(public_key)

# 2. Hash file
sha256 = hashlib.sha256()
with open(FILE_PATH, "rb") as f:
    for chunk in iter(lambda: f.read(1024 * 1024), b""):
        sha256.update(chunk)

file_hash = sha256.hexdigest()

# 3. Sign hash
signature = signing_key.sign(file_hash.encode()).signature.hex()

metadata = {
    "title": "AnthroHeart Master Archive",
    "file": os.path.basename(FILE_PATH),

    "hashes": {
        "sha256": hash_sha256,
        "sha512": hash_sha512
    },

    "author": AUTHOR,
    "organization": ORGANIZATION,

    "license": LICENSE,
    "rights": RIGHTS,
    "dedication": (
        "This work is dedicated to the public domain under CC0 1.0. "
        "No rights reserved."
    ),

    "publication_date": PUBLICATION_DATE,
    "jurisdiction": "Worldwide",

    "developed_on": "Windows 11",

    "website": WEBSITE,
    "donation_url": DONATION_URL,
    "metadata_reference": "https://metadata.anthroentertainment.com",

    "signatures": {
        "sha256": signature_sha256,
        "sha512": signature_sha512
    },

    "public_key": public_key_hex,
    "signing_algorithm": "Ed25519",

    "notes": [
        "Includes commissioned artwork with full copyright buyout (2019).",
        "Includes original literary and musical works.",
        "Released as public domain to support indie studios and creators.",
        "Trademark rights retained separately by Anthro Entertainment LLC."
    ]
}

with open(f"{OUT_DIR}/signature_metadata.json", "w") as f:
    json.dump(metadata, f, indent=2)

# 4. Load Arweave wallet
wallet = Wallet(WALLET_JSON)

# 5. Upload file
file_tx = Transaction(wallet, file_path=FILE_PATH)
file_tx.add_tag("Content-Type", "application/x-7z-compressed")
file_tx.add_tag("SHA256", file_hash)
file_tx.add_tag("Public-Key", public_key)
file_tx.add_tag("Signature", signature)
file_tx.sign()
file_tx.send()

# 6. Upload metadata
meta_tx = Transaction(wallet, data=json.dumps(metadata).encode())
meta_tx.add_tag("Content-Type", "application/json")
meta_tx.sign()
meta_tx.send()

print("\n✅ PERMANENT ARCHIVE COMPLETE")
print(f"File TX:     {file_tx.id}")
print(f"Metadata TX: {meta_tx.id}")
print(f"URL:         https://arweave.net/{file_tx.id}")
